#include <frost_utils.h>
#include <frost_utils_file.h>
#include <frost_utilsmanager.h>
#include <iostream>

using namespace Frost;

void LogCout(const s_str& sMessage, const s_bool& bStamps, const s_uint& uiOffset)
{
    std::cout << sMessage.Get();
}

s_bool IsNumber(const s_str& s)
{
    s_str t = s;
    t.Trim(' ');
    if (t.StartsWith("+") || t.StartsWith("-"))
        t.EraseFromStart(1);

    s_bool bCommaFound;
    s_str::iterator iter = t.Begin();
    while (iter != t.End())
    {
        if (*iter == '.' || *iter == ',')
        {
            if (!bCommaFound) bCommaFound = true;
            else return false;
        }
        else if (!s_str::IsNumber(*iter))
            break;

        ++iter;
    }

    if (iter == t.End())
        return true;

    if (*iter != 'e')
        return false;

    ++iter;
    if (iter == t.End())
        return false;

    if (*iter == '+' || *iter == '-')
    {
        ++iter;
        if (iter == t.End())
            return false;
    }

    while (iter != t.End())
    {
        if (!s_str::IsNumber(*iter))
            return false;

        ++iter;
    }

    return true;
}

struct TimeKey
{
    s_int  iSecond;
    s_int  iMilisecond;

    TimeKey() : iSecond(s_int::NaN), iMilisecond(s_int::NaN)
    {
    }

    TimeKey(const s_float& fSec) : iSecond(s_float::RoundDown(fSec)), iMilisecond((fSec - s_float::RoundDown(fSec))*1000)
    {

    }

    explicit TimeKey(const s_str& sStr)
    {
        s_bool bInvalid = true;

        s_ctnr<s_str> lWords = sStr.Cut(":");
        if (lWords.GetSize() == 3)
        {
            s_ctnr<s_str>::iterator iter = lWords.Begin();

            iSecond = s_int(*iter)*3600;
            ++iter;
            iSecond += s_int(*iter)*60;
            ++iter;
            lWords = iter->Cut(",");
            iSecond += s_int(lWords.Front());

            if (lWords.GetSize() == 2)
                iMilisecond = s_int(lWords.Back());

            bInvalid = false;
        }
        else if (lWords.GetSize() == 2)
        {
            s_ctnr<s_str>::iterator iter = lWords.Begin();

            iSecond = s_int(*iter)*60;
            ++iter;
            lWords = iter->Cut(",");
            iSecond += s_int(lWords.Front());

            if (lWords.GetSize() == 2)
                iMilisecond = s_int(lWords.Back());

            bInvalid = false;
        }
        else if (lWords.GetSize() == 1)
        {
            s_ctnr<s_str>::iterator iter = lWords.Begin();

            lWords = iter->Cut(",");
            if (IsNumber(lWords.Front()))
            {
                iSecond = s_int(lWords.Front());

                if (lWords.GetSize() == 2)
                    iMilisecond = s_int(lWords.Back());

                bInvalid = false;
            }
        }

        if (bInvalid)
        {
            iSecond = s_int::NaN;
            iMilisecond = s_int::NaN;
        }
    }

    TimeKey(const s_int& iSec, const s_int& iMili) : iSecond(iSec), iMilisecond(iMili)
    {
        while (iMilisecond >= 1000)
        {
            ++iSecond;
            iMilisecond -= 1000;
        }
        while (iMilisecond < 0)
        {
            --iSecond;
            iMilisecond += 1000;
        }
    }

    s_bool IsValid() const
    {
        return iSecond.IsValid() && iMilisecond.IsValid();
    }

    s_str ToString() const
    {
        s_int iHour = iSecond/3600;
        s_int iMin = (iSecond - iHour*3600)/60;
        s_int iSec = (iSecond - iHour*3600 - iMin*60);

        return s_str::Convert(iHour, 2)+":"+s_str::Convert(iMin, 2)+":"+s_str::Convert(iSec, 2)+","+s_str::Convert(iMilisecond, 3);
    }

    TimeKey operator + (const TimeKey& mTime) const
    {
        return TimeKey(iSecond + mTime.iSecond, iMilisecond + mTime.iMilisecond);
    }

    TimeKey& operator += (const TimeKey& mTime)
    {
        iSecond += mTime.iSecond;
        iMilisecond += mTime.iMilisecond;

        while (iMilisecond >= 1000)
        {
            ++iSecond;
            iMilisecond -= 1000;
        }
        while (iMilisecond < 0)
        {
            --iSecond;
            iMilisecond += 1000;
        }

        return *this;
    }

    TimeKey operator - (const TimeKey& mTime) const
    {
        return TimeKey(iSecond - mTime.iSecond, iMilisecond - mTime.iMilisecond);
    }

    TimeKey& operator -= (const TimeKey& mTime)
    {
        iSecond -= mTime.iSecond;
        iMilisecond -= mTime.iMilisecond;

        while (iMilisecond >= 1000)
        {
            ++iSecond;
            iMilisecond -= 1000;
        }
        while (iMilisecond < 0)
        {
            --iSecond;
            iMilisecond += 1000;
        }

        return *this;
    }

    s_bool operator < (const TimeKey& mTime) const
    {
        if (iSecond == mTime.iSecond)
            return (iMilisecond < mTime.iMilisecond);
        else
            return (iSecond < mTime.iSecond);
    }

    s_bool operator <= (const TimeKey& mTime) const
    {
        if (iSecond == mTime.iSecond)
            return (iMilisecond <= mTime.iMilisecond);
        else
            return (iSecond <= mTime.iSecond);
    }

    s_bool operator > (const TimeKey& mTime) const
    {
        if (iSecond == mTime.iSecond)
            return (iMilisecond > mTime.iMilisecond);
        else
            return (iSecond > mTime.iSecond);
    }

    s_bool operator >= (const TimeKey& mTime) const
    {
        if (iSecond == mTime.iSecond)
            return (iMilisecond >= mTime.iMilisecond);
        else
            return (iSecond >= mTime.iSecond);
    }
};

struct Entry
{
    s_uint uiID;
    TimeKey mStart;
    TimeKey mEnd;
    s_str sContent;

    s_str ToString() const
    {
        s_str sStr;
        sStr += s_str::Convert(uiID) + "\n";
        sStr += mStart.ToString() + " --> " + mEnd.ToString() + "\n";
        sStr += sContent;
        return sStr;
    }
};

s_bool FindNext(s_ctnr<Entry>& array, s_ctnr<Entry>::iterator& iter, const s_str& str)
{
    while (iter != array.end())
    {
        if (iter->sContent.Find(str))
            return true;

        ++iter;
    }

    return false;
}

s_bool FindPrevious(s_ctnr<Entry>& array, s_ctnr<Entry>::iterator& iter, const s_str& str)
{
    while (iter != array.begin())
    {
        --iter;

        if (iter->sContent.Find(str))
            return true;
    }

    return false;
}

void DisplayHelp()
{
    Log("\nHelp :\n\n");
    Log("  This program can do some basic editing on subtitles.\n");
    Log("  Its only function for now it to be able to add a delay to a\n");
    Log("  particular sentence, shifting all the following ones.\n");
    Log("  To do so, you must first pick the sentence you want to shift\n");
    Log("  (using any of the various possibilities discribed below),\n");
    Log("  then press 'Enter' to input the amount of seconds you want\n");
    Log("  to add/remove. The program will update the subtitle accordingly\n");
    Log("  and save it right away, so you can immediately check the result\n");
    Log("  in your favorite movie player.\n");
    Log("  Note that some players require a restart for the changes to be\n");
    Log("  effectives (VLC for example).\n\n");
    Log("Available search commands :\n\n");
    Log("  hh:mm:ss,mili : selects the sentence just after the provided time stamp\n");
    Log("  any text      : searches for occurences of 'any text', enters search mode\n");
    Log("  ?             : exits search mode\n");
    Log("  #nn           : selects the 'nn'th sentence ('nn'th occurence in search mode)\n");
    Log("  +x            : advances 'x' times (next occurences in search mode)\n");
    Log("  +             : advances once (next occurence in search mode)\n");
    Log("  -x            : steps back 'x' times (previous occurences in search mode)\n");
    Log("  -             : steps back once (previous occurence in search mode)\n\n");
    Log("Other commands :\n\n");
    Log("  'empty'       : starts editing the previous sentence's time stamp\n");
    Log("  help or h     : displays this text\n");
    Log("  quit or q     : exits the program\n\n");
}

int main(int argc, char* argv[])
{
    UtilsManager::GetSingleton()->SetLogFunction(&LogCout);

    s_ctnr<Entry> lEntryList;
    s_str sFileName;

    if (argc > 1)
    {
        sFileName = argv[argc-1];

        File mSub(sFileName, File::I);

        if (!mSub.IsValid())
        {
            Log("Cannot open file : "+sFileName+".\n");
            return 1;
        }

        Entry mEntry;
        s_uint uiCount;
        s_uint uiLineNbr;
        while (mSub.IsValid())
        {
            ++uiLineNbr;
            s_str sLine = mSub.GetLine();

            if (!sLine.IsEmpty())
            {
                if (uiCount == 0)
                    mEntry.uiID = s_uint(sLine);
                else if (uiCount == 1)
                {
                    s_ctnr<s_str> lWords = sLine.Cut(" --> ");
                    if (lWords.GetSize() == 2)
                    {
                        mEntry.mStart = TimeKey(lWords.Front());
                        mEntry.mEnd = TimeKey(lWords.Back());
                    }
                    else
                    {
                        Error(s_str::Convert(uiLineNbr), "Bad time tag.\n");
                        return 1;
                    }
                }
                else
                {
                    mEntry.sContent += sLine + "\n";
                }
                ++uiCount;
            }
            else if (uiCount != 0)
            {
                lEntryList.PushBack(mEntry);
                mEntry.sContent = "";
                uiCount = 0;
            }

        }

        mSub.Close();

        Log("Subtitle successfully loaded !\n");
    }
    else
    {
        Log("Please provide a file name or drag and drop the file on the executable.\n");
        return 0;
    }

    Log("If you need help, type 'help' or 'h'. Type 'q' to exit.\n\n");

    s_bool bNoDisplay = true;
    s_bool bSearchMode = false;
    s_str  sSearch = "";
    TimeKey mKey;
    s_ctnr<Entry>::iterator iter = lEntryList.Begin();

    while (true)
    {
        if (!bNoDisplay)
        {
            if (iter != lEntryList.End())
                Log("\n["+s_ptrdiff(iter - lEntryList.Begin())+"] "+iter->mStart.ToString()+" :\n\n"+iter->sContent+"\n");
        }

        bNoDisplay = false;

        Log("> ");

        s_str s;
        getline(std::cin, s.Get());
        s_str sLow = s; sLow.ToLower();

        if (s.IsEmpty())
        {
            bNoDisplay = true;
            Log("\nHow many seconds do you want to add / remove (empty to abord) ? ");

            s_float fSec = s_float::NaN;
            s_bool bFirst = true;
            while (!fSec.IsValid())
            {
                if (!bFirst)
                    Log("Invalid time duration, please enter a floating point number (or enter nothing to abord) : ");

                std::string s_; s_str s;
                getline(std::cin, s_); s = s_;

                if (s.IsEmpty())
                {
                    break;
                }
                else if (s.StartsWith("?") || s.StartsWith("#") || s.Find(":"))
                {
                    bFirst = false;
                    continue;
                }

                fSec = s_float(s_str(s));
                bFirst = false;
            }

            if (!fSec.IsValid())
            {
                Log("\n");
                continue;
            }

            Log("Editing subtitle, please wait... ");

            s_uint uiCount;
            while (iter != lEntryList.End())
            {
                iter->mStart += fSec;
                iter->mEnd += fSec;
                ++iter;
                ++uiCount;
            }

            Log("done ("+uiCount+" entries modified).\nSaving... ");

            File mSub(sFileName, File::O);

            foreach (iter, lEntryList)
            {
                mSub.WriteLine(iter->ToString());
            }

            mSub.Close();

            Log(" done.\n");
            continue;
        }

        if (sLow == "q" || sLow == "quit")
            return 1;

        if (sLow == "h" || sLow == "help")
        {
            DisplayHelp();
            bNoDisplay = true;
            continue;
        }

        char c = s[0];

        if (c == '#')
        {
            s.EraseFromStart(1);
            s_uint uiNum = s_uint(s);

            if (!bSearchMode)
            {
                s_ctnr<Entry>::iterator iterOld = iter;
                iter = lEntryList.Begin();
                s_uint uiIter = uiNum;
                while (iter != lEntryList.End() && uiIter != 0)
                {
                    --uiIter;
                    ++iter;
                }

                if (iter == lEntryList.End())
                {
                    Log("Warning : not so many phrases (max : "+(lEntryList.GetSize()-1)+").\n\n");
                    bNoDisplay = true;
                    iter = iterOld;
                    continue;
                }

                mKey = iter->mStart;
            }
            else
            {
                ++uiNum;
                s_ctnr<Entry>::iterator iterOld = iter;
                iter = lEntryList.Begin();
                s_bool bResult = FindNext(lEntryList, iter, sSearch);
                --uiNum;

                while (bResult && uiNum != 0)
                {
                    iterOld = iter;
                    ++iter;
                    bResult = FindNext(lEntryList, iter, sSearch);
                    --uiNum;
                }

                if (!bResult)
                {
                    Log("Warning : no further matches, displaying last one.\n\n");
                    iter = iterOld;
                }

                mKey = iter->mStart;
            }
        }
        else if (c == '?')
        {
            sSearch = s;
            sSearch.EraseFromStart(1);

            if (sSearch.IsEmpty())
            {
                bSearchMode = false;
                bNoDisplay = true;
                continue;
            }

            s_ctnr<Entry>::iterator iterOld = iter;
            s_bool bResult = FindNext(lEntryList, iter, sSearch);
            if (!bResult)
            {
                iter = lEntryList.Begin();
                bResult = FindNext(lEntryList, iter, sSearch);

                if (!bResult)
                {
                    Log("Warning : no match for '"+sSearch+"'.\n\n");
                    bNoDisplay = true;
                    iter = iterOld;
                    mKey = iter->mStart;
                    continue;
                }
            }

            bSearchMode = true;
        }
        else if (c == '+')
        {
            s_uint uiNum;

            if (s.GetSize() == 1)
                uiNum = 1;
            else
            {
                s.EraseFromStart(1);
                uiNum = s_uint::Max(1, s_uint(s));
            }

            if (!bSearchMode)
            {
                s_uint uiIter = uiNum;
                while (iter != lEntryList.End() && uiIter != 0)
                {
                    ++iter;
                    --uiIter;
                }

                if (iter == lEntryList.End())
                {
                    --iter;
                    if (uiNum == 1)
                    {
                        Log("Warning : no further phrases.\n\n");
                        bNoDisplay = true;
                    }
                    else
                    {
                        Log("Warning : only "+(uiNum-uiIter)+" further phrases, displaying last one.\n\n");
                        mKey = iter->mStart;
                    }

                    continue;
                }

                mKey = iter->mStart;
            }
            else
            {
                s_ctnr<Entry>::iterator iterOld = iter;
                s_bool bResult = true;
                s_uint uiIter = uiNum;
                while (bResult && uiIter != 0)
                {
                    iterOld = iter;
                    ++iter;
                    bResult = FindNext(lEntryList, iter, sSearch);
                    --uiIter;
                }

                if (!bResult)
                {
                    iter = iterOld;

                    if (uiNum == 1)
                    {
                        Log("Warning : no further matches.\n\n");
                        bNoDisplay = true;
                        continue;
                    }
                    else
                    {
                        Log("Warning : only "+(uiNum-uiIter)+" further matches, displaying last one.\n\n");
                    }
                }

                mKey = iter->mStart;
            }
        }
        else if (c == '-')
        {
            s_uint uiNum;

            if (s.GetSize() == 1)
                uiNum = 1;
            else
            {
                s.EraseFromStart(1);
                uiNum = s_uint::Max(1, s_uint(s));
            }

            if (!bSearchMode)
            {
                s_ctnr<Entry>::iterator iterOld = iter;
                s_uint uiIter = uiNum;
                while (iter != lEntryList.Begin() && uiIter != 0)
                {
                    --iter;
                    --uiIter;
                }

                if (iter == lEntryList.Begin() && uiIter != 0)
                {
                    iter = iterOld;

                    if (uiNum == 1)
                    {
                        Log("Warning : no phrases before this point.\n\n");
                        bNoDisplay = true;
                        continue;
                    }
                    else
                    {
                        Log("Warning : only "+(uiNum-uiIter)+" matches before this point, displaying last one.\n\n");
                    }
                }

                mKey = iter->mStart;
            }
            else
            {
                s_ctnr<Entry>::iterator iterOld = iter;
                s_bool bResult = true;
                s_uint uiIter = uiNum;
                while (bResult && uiIter != 0)
                {
                    iterOld = iter;
                    bResult = FindPrevious(lEntryList, iter, sSearch);
                    --uiIter;
                }

                if (!bResult)
                {
                    if (uiNum == 1)
                    {
                        Log("Warning : no matches before this point.\n\n");
                        bNoDisplay = true;
                        iter = iterOld;
                    }
                    else
                    {
                        Log("Warning : only "+(uiNum-uiIter)+" phrases before this point, displaying last one.\n\n");
                        mKey = iter->mStart;
                    }

                    continue;
                }
            }
        }
        else
        {
            TimeKey mTempKey = TimeKey(s_str(s));
            if (mTempKey.IsValid())
            {
                s_ctnr<Entry>::iterator iterOld = iter;

                foreach (iter, lEntryList)
                {
                    if (iter->mStart >= mTempKey)
                        break;
                }

                if (iter == lEntryList.End())
                {
                    Log("Warning : no phrase after "+mTempKey.ToString()+".\n\n");
                    bNoDisplay = true;
                    iter = iterOld;
                    continue;
                }

                mKey = iter->mStart;
            }
            else
            {
                sSearch = s;

                s_ctnr<Entry>::iterator iterOld = iter;
                s_bool bResult = FindNext(lEntryList, iter, sSearch);
                if (!bResult)
                {
                    iter = lEntryList.Begin();
                    bResult = FindNext(lEntryList, iter, sSearch);

                    if (!bResult)
                    {
                        Log("Warning : no match for '"+sSearch+"'.\n\n");
                        bNoDisplay = true;
                        iter = iterOld;
                        mKey = iter->mStart;
                        continue;
                    }
                }

                bSearchMode = true;
            }
        }
    }

    return 0;
}
